/*
 * Copyright (c) 2024 Sascha Wildner <swildner@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>
#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cb2c.h"
#include "basic20_parse.h"

static void emit(bool, const char *, ...) __printflike(2, 3);
static void generate_CLR(void);
static void generate_DEF(void);
static void generate(node_t *);
static void indent(void);
static void output_str(char *str);

static int indent_level, return_lbl_index;
static int FOR_count, IF_count;

static void
declare_data(void)
{
	int i;
	char srcval[4096], destval[4096];

	emit(true, "static char *_cb2crt_data_values[%d] = {\n", data_index);
	indent_level++;
	for (i = 0; i < data_index; i++) {
		sprintf(srcval, "%s%s%s",
		    data_values[i][0] != '"' ? "\"" : "",
		    data_values[i],
		    (data_values[i][0] != '"' ||
			data_values[i][strlen(data_values[i]) - 1] != '"') ?
		    "\"" : "");
		STRING_convert(srcval, destval, strlen(srcval));
		indent();
		output_str(destval);
		if (i < data_index - 1)
			emit(false, ",");
		emit(false, "\n");
	}
	indent_level--;
	emit(true, "};\nstatic int32_t _cb2crt_data_index;\n\n");
}

static void
declare_vars(void)
{
	size_t i;
	int dimn;
	node_t *np;
	symbol_t *sp;

	for (i = 0; i < symtbl->st_capacity; i++) {
		for (sp = symtbl->st_table[i]; sp != NULL; sp = sp->s_next) {
			switch (sp->s_btype) {
			case SBTYPE_FN:
				emit(false, "static int32_t (*%s)(int32_t);\n",
				    sp->s_name);
				/* FALLTHROUGH */
			case SBTYPE_DEF:
				continue;
				break;
			case SBTYPE_INT:
				emit(false, "static ");
				if (sp->s_auto_ctype) {
					switch (sp->s_ctype_num) {
					case SCTYPE_U8:
						emit(false, "uint8_t");
						break;
					case SCTYPE_S8:
						emit(false, "int8_t");
						break;
					case SCTYPE_U16:
						emit(false, "uint16_t");
						break;
					case SCTYPE_S16:
						emit(false, "int16_t");
						break;
					case SCTYPE_U32:
						emit(false, "uint32_t");
						break;
					case SCTYPE_S32:
					default:
						emit(false, "int32_t");
						break;
					}
				} else {
					emit(false, "int32_t");
				}
				emit(false, " ");
				break;
			case SBTYPE_REAL:
				emit(false, "static float ");
				break;
			case SBTYPE_RETURN:
				emit(false, "static void *%s;\n", sp->s_name);
				continue;
				break;
			case SBTYPE_STRING:
				emit(false, "static char ");
				break;
			default:
				break;
			}
			emit(false, "%s", sp->s_name);
			if (sp->s_np != NULL) {
				for (np = sp->s_np, dimn = 0; np != NULL;
				     np = np->n_next, dimn++) {
					emit(false, "[%d]",
					    sp->s_ctype_dimns[dimn] < 11 ?
					    11 : sp->s_ctype_dimns[dimn] + 1);
				}
			}
			if (sp->s_btype == SBTYPE_STRING) {
				if (sp->s_auto_ctype &&
				    sp->s_ctype_strlen != 0) {
					emit(false, "[%zu]",
					    sp->s_ctype_strlen);
				} else {
					emit(false, "[256]");
				}
			}
			emit(false, ";\n");
		}
	}
}

static void
emit(bool autoindent, const char *fmt, ...)
{
	va_list ap;

	if (autoindent)
		indent();
	va_start(ap, fmt);
	vfprintf(stdout, fmt, ap);
	va_end(ap);
}

static void
global(void)
{
	emit(true, "/*\n");
	emit(true, " * translated with cb2c %s\n", CB2C_VERSION);
#if 0
	if (SEEN_MATH)
		emit(true, " * build with -lm\n");
#endif
	if (SEEN_TOK(TOK_PRINT) && rflag)
		emit(true, " * build with -lprintf_flt\n");
	emit(true, " */\n\n");
	emit(true, "#pragma clang diagnostic ignored "
	    "\"-Wchar-subscripts\"\n");
	emit(true, "#pragma clang diagnostic ignored "
	    "\"-Winvalid-source-encoding\"\n");
	emit(true, "#pragma clang diagnostic ignored "
	    "\"-Wtautological-constant-out-of-range-compare\"\n\n");
	emit(true, "#include <cbm.h>\n");
	if (SEEN_TOK(TOK_INPUT) || SEEN_TOK(TOK_PRINT))
		emit(true, "#include <stdio.h>\n");
	if (SEEN_TOK(TOK_FRE) || SEEN_TOK(TOK_RND))
		emit(true, "#include <limits.h>\n");
	emit(true, "#include <ctype.h>\n");
	if (!rflag)
		emit(true, "#include <inttypes.h>\n");
#if 0
	if (SEEN_MATH)
		emit(true, "#include <math.h>\n");
#endif
	emit(true, "#include <stdbool.h>\n");
	emit(true, "#include <stdlib.h>\n");
	emit(true, "#include <string.h>\n\n");
	emit(true, "#include \"cb2c_rt.c\"\n\n");
	if (SEEN_LMR)
		emit(true, "static char _cb2crt_dummybuf[256];\n");
	else if (SEEN_TOK(TOK_INPUT))
		emit(true, "static __unused char _cb2crt_dummybuf[79];\n");
	if (SEEN_TOK(TOK_SAVE))
		emit(true, "extern char __bss_end;\n\n");
	if (SEEN_TOK(TOK_DATA))
		declare_data();
	if (symtbl->st_size > 0) {
		declare_vars();
		emit(true, "\n");
	}
	if (SEEN_TOK(TOK_CLR) || SEEN_TOK(TOK_LOAD) || SEEN_TOK(TOK_RUN))
		generate_CLR();
	generate_DEF();
}

static void
indent(void)
{
	int i;

	for (i = 0; i < indent_level; i++)
		putchar('\t');
}

static void
main1(void)
{
	emit(true, "int\nmain(void)\n{\n");
	indent_level++;
	emit(true, "VIC.addr &= 0xfd;\t/* upper case */\n");
}

static void
main2(void)
{
	while (IF_count > 0) {
		indent_level--;
		emit(true, "}\n");
		IF_count--;
	}
	while (FOR_count > 0) {
		emit(true, "break;\n");
		indent_level--;
		emit(true, "}\n");
		FOR_count--;
	}
	emit(true, "exit(EXIT_SUCCESS);\n");
	indent_level--;
	emit(true, "}\n");
}

static void
output_str(char *str)
{
	unsigned char *ucp;

	emit(false, "\"");
	for (ucp = (unsigned char *)str; *ucp != '\0'; ucp++)
		emit(false, isprint(*ucp) ? "%c" : "\\%03o", *ucp);
	emit(false, "\"");
}

static void
generate_CLR(void)
{
	size_t i;
	symbol_t *sp;

	/*
	 * FIXME:
	 * delete the state of loops (FOR and NEXT)
	 */
	emit(true, "static void\n");
	emit(true, "clr(void)\n");
	emit(true, "{\n");
	indent_level++;
	if (SEEN_TOK(TOK_OPEN))
		emit(true, "int i;\n\n");
	if (SEEN_TOK(TOK_DATA))
		emit(true, "_cb2crt_data_index = 0;\n");
	for (i = 0; i < symtbl->st_capacity; i++) {
		for (sp = symtbl->st_table[i]; sp != NULL; sp = sp->s_next) {
			switch (sp->s_btype) {
			case SBTYPE_INT:
			case SBTYPE_REAL:
			case SBTYPE_STRING:
				emit(true, "memset(%s%s, 0, sizeof(%s));\n",
				    (sp->s_ctype_dimns[0] == 0 &&
					sp->s_btype != SBTYPE_STRING ?
					"&" : ""),
				    sp->s_name, sp->s_name);
				break;
			case SBTYPE_FN:
			case SBTYPE_RETURN:
				emit(true, "%s = NULL;\n", sp->s_name);
				break;
			default:
				break;
			}
		}
	}
	if (SEEN_TOK(TOK_OPEN)) {
		emit(true, "for (i = 1; i <= 255; i++)\n");
		indent_level++;
		emit(true, "cbm_k_close(i);\n");
		indent_level--;
	}
	indent_level--;
	emit(true, "}\n\n");
}

static void
generate_DEF(void)
{
	size_t i;
	symbol_t *sp;

	for (i = 0; i < symtbl->st_capacity; i++) {
		for (sp = symtbl->st_table[i]; sp != NULL; sp = sp->s_next) {
			if (sp->s_btype == SBTYPE_DEF) {
				emit(true, "static %s\n",
				    rflag ? "float" : "int32_t");
				emit(true, "%s(__unused %s %s)\n", sp->s_name,
				    rflag ? "float" : "int32_t", sp->s_fnarg);
				emit(true, "{\n");
				indent_level++;
				emit(true, "return (");
				generate(sp->s_np);
				emit(false, ");\n");
				indent_level--;
				emit(true, "}\n\n");
			}
		}
	}
}

static void
generate(node_t *n)
{
	switch (n->n_tok_type) {
	case TOK_ABS:
		emit(false, "labs((int32_t)(");
		generate(n->n_left);
		emit(false, "))");
		break;
	case TOK_AND:
		if (n->n_val_type == VTYPE_NUM)
			emit(false, "(int32_t)(");
		generate(n->n_left);
		if (n->n_val_type == VTYPE_BOOL)
			emit(false, " && ");
		else if (n->n_val_type == VTYPE_NUM)
			emit(false, ") & (int32_t)(");
		generate(n->n_mid);
		if (n->n_val_type == VTYPE_BOOL)
			emit(false, " ? -1L : 0L");
		else if (n->n_val_type == VTYPE_NUM)
			emit(false, ")");
		break;
	case TOK_ARROWUP:
		emit(false, "_cb2crt_power(");
		generate(n->n_left);
		emit(false, ", ");
		generate(n->n_mid);
		emit(false, ")");
		break;
	case TOK_ASC:
		generate(n->n_left);
		emit(false, "[0]");
		break;
	case TOK_ASTERISK:
		emit(false, "(");
		generate(n->n_left);
		emit(false, ") * (");
		generate(n->n_mid);
		emit(false, ")");
		break;
	case TOK_ATN:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "atanf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped ATN statement */");
#endif
		break;
	case TOK_CHR_D:
		generate(n->n_left);
		break;
	case TOK_CLOSE:
		emit(true, "cbm_k_close(");
		generate(n->n_left);
		emit(false, ");\n");
		break;
	case TOK_CLR:
		emit(true, "clr();\n");
		break;
	case TOK_CMD:
		break;
	case TOK_CONT:
		/* assume CONT's purpose is to just wait (infinite loop) */
		emit(true, "for (;;);\n");
		break;
	case TOK_COS:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "cosf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped COS statement */");
#endif
		break;
	case TOK_DATA:
		/* nothing to do here, see the scanner and declare_data() */
		break;
	case TOK_DEF:
		emit(true, "%s = %s_%p;\n", n->n_str, n->n_str, n->n_left);
		break;
	case TOK_DIM:
		/* nothing to do here, see parser */
		break;
	case TOK_END:
		emit(true, "exit(EXIT_SUCCESS);\n");
		break;
	case TOK_EQ:
	{
		if (n->n_right == NULL) {
			enum val_type l_vtype = n->n_left->n_val_type;
			enum val_type m_vtype = n->n_mid->n_val_type;
			if (l_vtype == VTYPE_NUM || m_vtype == VTYPE_CHAR) {
				/*
				 * assign to a numeric variable. this includes
				 * assigning a chr$() to a string.
				 */
				indent();
				generate(n->n_left);
				if (n->n_mid->n_val_type == VTYPE_CHAR)
					emit(false, "[0]");
				emit(false, " = (");
				generate(n->n_mid);
				emit(false, ");\n");
			} else {
				/*
				 * assign to a string variable. ti$ needs
				 * special treatment.
				 */
				emit(true, "strcpy(");
				if (strcmp(n->n_left->n_str, "s_ti") == 0)
					emit(false, "s_ti");
				else
					generate(n->n_left);
				emit(false, ", ");
				generate(n->n_mid);
				emit(false, ");\n");
				if (strcmp(n->n_left->n_str, "s_ti") == 0) {
					emit(true,
					    "_cb2crt_set_clock_from_ti_d"
					    "(s_ti);\n");
				}
			}
		} else {
			/* expression */
			enum val_type l_vtype = n->n_left->n_val_type;
			enum val_type r_vtype = n->n_right->n_val_type;
			emit(false, "(");
			if (l_vtype == VTYPE_STRING && r_vtype == VTYPE_STRING)
				emit(false, "strcmp(");
			else
				emit(false, "(");
			generate(n->n_left);
			if (l_vtype == VTYPE_STRING && r_vtype != VTYPE_STRING)
				emit(false, "[0]");
			if (l_vtype == VTYPE_STRING && r_vtype == VTYPE_STRING)
				emit(false, ", ");
			else
				emit(false, ") == (");
			generate(n->n_right);
			if (l_vtype != VTYPE_STRING && r_vtype == VTYPE_STRING)
				emit(false, "[0]");
			if (l_vtype == VTYPE_STRING && r_vtype == VTYPE_STRING)
				emit(false, ") == 0");
			else
				emit(false, ")");
			emit(false, " ? -1L : 0L)");
		}
		break;
	}
	case TOK_EXP:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "expf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped EXP statement */");
#endif
		break;
	case TOK_FN:
		emit(false, "fn%s(", n->n_str);
		generate(n->n_left);
		emit(false, ")");
		break;
	case TOK_FOR:
		/*
		 * If we encounter either a "FORI=1TO1000:NEXT" or a
		 * "FORI=1TO1000:NEXTI", assume it is an empty loop
		 * to cause a delay.
		 */
		if (n->n_next != NULL && n->n_next->n_tok_type == TOK_NEXT &&
		    (n->n_next->n_left == NULL ||
			(strcmp(n->n_next->n_left->n_str, n->n_str) == 0 &&
			    n->n_next->n_left->n_next == NULL))) {
			emit(true, "_cb2crt_delay(((");
			generate(n->n_mid);
			emit(false, ") + 1 - (");
			generate(n->n_left);
			emit(false, ")) / %s16.6);\n",
			    rflag ? "" : "(int32_t)");
			n->n_next->n_num = 1;
			break;
		}
		emit(true, "for (%s = (", n->n_str);
		generate(n->n_left);
		emit(false, "), %s_cond = (", n->n_str);
		if (n->n_mid->n_tok_type == TOK_ID &&
		    strcmp(n->n_str, n->n_mid->n_str) == 0) {
			generate(n->n_left);
		} else {
			generate(n->n_mid);
		}
		emit(false, "); \n");
		emit(true, "     (");
		if (n->n_right != NULL) {
			generate(n->n_right);
			emit(false,
			    ") < 0 ? %s >= %s_cond : %s <= %s_cond;\n",
			    n->n_str, n->n_str, n->n_str, n->n_str);
			emit(true, "     %s += ", n->n_str);
			generate(n->n_right);
			emit(false, ") {\n");
		} else {
			generate(n->n_left);
			emit(false, ") > %s_cond ? %s <= (",
			    n->n_str, n->n_str);
			generate(n->n_left);
			emit(false, ") : %s <= %s_cond;\n",
			    n->n_str, n->n_str);
			emit(true, "     %s++) {\n", n->n_str);
		}
		FOR_count++;
		indent_level++;
		break;
	case TOK_FRE:
		emit(false, "(int32_t)(INT_MAX)");
		break;
	case TOK_GET:
	{
		node_t *np;
		if (n->n_mid != NULL) {
			/* GET# */
			emit(true, "cbm_k_chkin(");
			generate(n->n_mid);
			emit(false, ");\n");
		}
		for (np = n->n_left; np != NULL; np = np->n_next) {
			indent();
			generate(np);
			if (np->n_val_type == VTYPE_STRING)
				emit(false, "[0]");
			if (n->n_mid != NULL)
				emit(false, " = cbm_k_basin();\n");
			else
				emit(false, " = tolower(cbm_k_getin());\n");
			if (np->n_val_type == VTYPE_STRING) {
				indent();
				generate(np);
				emit(false, "[1] = '\\0';\n");
			}
		}
		break;
	}
	case TOK_GOSUB:
	{
		node_t *return_stmt;
		if (!LINE_EXISTS(n->n_num)) {
			emit(true, "/* no such line: gosub %d */\n",
			    n->n_num);
			break;
		}
		return_stmt = ptree_find_token(line_ary[n->n_num], TOK_RETURN);
		if (return_stmt != NULL) {
			emit(true, "__RETURN%d_target = &&return_label%d;\n",
			    return_stmt->n_lineno, return_lbl_index + 1);
		}
		emit(true, "goto line_label%d;\n", n->n_num);
		if (return_stmt != NULL)
			emit(false, "return_label%d:;\n", ++return_lbl_index);
		break;
	}
	case TOK_GOTO:
		if (!LINE_EXISTS(n->n_num))
			emit(true, "/* no such line: goto %d */\n", n->n_num);
		else
			emit(true, "goto line_label%d;\n", n->n_num);
		break;
	case TOK_GT:
	{
		enum val_type l_vtype = n->n_left->n_val_type;
		enum val_type m_vtype = n->n_mid->n_val_type;
		emit(false, "(");
		if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING)
			emit(false, "strcmp(");
		else
			emit(false, "(");
		generate(n->n_left);
		if (l_vtype == VTYPE_STRING && m_vtype != VTYPE_STRING)
			emit(false, "[0]");
		if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING) {
			emit(false, ", ");
		} else {
			if (n->n_num == TOK_EQ)
				emit(false, ") >= (");
			else
				emit(false, ") > (");
		}
		generate(n->n_mid);
		if (l_vtype != VTYPE_STRING && m_vtype == VTYPE_STRING)
			emit(false, "[0]");
		if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING) {
			if (n->n_num == TOK_EQ)
				emit(false, ") >= 0");
			else
				emit(false, ") > 0");
		} else {
			emit(false, ")");
		}
		emit(false, " ? -1L : 0L)");
		break;
	}
	case TOK_ID:
	{
		node_t *np;
		/* special variables ST, TI, and TI$ */
		if (strcmp(n->n_str, rflag ? "r_st" : "I_st") == 0 &&
		    n->n_left == NULL) {
			emit(false, "_cb2crt_get_st()");
			break;
		} else if (strcmp(n->n_str, rflag ? "r_ti" : "I_ti") == 0 &&
		    n->n_left == NULL) {
			emit(false, "_cb2crt_get_ti()");
			break;
		} else if (strcmp(n->n_str, "s_ti") == 0) {
			emit(false, "_cb2crt_get_ti_d(s_ti)");
			break;
		}
		emit(false, "%s", n->n_str);
		if (n->n_left != NULL) {
			for (np = n->n_left; np != NULL; np = np->n_next) {
				emit(false, "[");
				generate(np);
				emit(false, "]");
			}
		}
		break;
	}
	case TOK_IF:
		emit(true, "if (");
		generate(n->n_left);
		if (n->n_left->n_val_type == VTYPE_STRING &&
		    n->n_left->n_next == NULL) {
			emit(false, "[0] != '\\0'");
		}
		if (n->n_num != -1) {
			emit(false, ") {\n");
			indent_level++;
			if (!LINE_EXISTS(n->n_num)) {
				emit(true, "/* no such line: goto %d */\n",
				    n->n_num);
			} else {
				emit(true, "goto line_label%d;\n", n->n_num);
			}
			indent_level--;
			emit(true, "}\n");
		} else if (n->n_mid != NULL &&
		    n->n_mid->n_tok_type == TOK_INTEGER) {
			emit(false, ") {\n");
			indent_level++;
			if (!LINE_EXISTS(n->n_mid->n_num)) {
				emit(true, "/* no such line: then %d */\n",
				    n->n_mid->n_num);
			} else {
				emit(true, "goto line_label%d;\n",
				    n->n_mid->n_num);
			}
			indent_level--;
			emit(true, "}\n");
		} else {
			emit(false, ") {\n");
			IF_count++;
			indent_level++;
			if (n->n_mid != NULL)
				generate(n->n_mid);
		}
		break;
	case TOK_INPUT:
	{
		node_t *np;
		if (n->n_str != NULL) {
			emit(true, "fputs(");
			output_str(n->n_str);
			emit(false, ", stdout);\n");
		}
		if (n->n_left != NULL) {
			for (np = n->n_left; np != NULL; np = np->n_next) {
				emit(true, "putchar('?');\n");
				if (np != n->n_left)
					emit(true, "putchar('?');\n");
				emit(true, "putchar(' ');\n");
				emit(true, "fgets(");
				if (np->n_val_type == VTYPE_STRING)
					generate(np);
				else
					emit(false, "_cb2crt_dummybuf");
				emit(false, ", 79, stdin);\n");
				if (np->n_val_type == VTYPE_STRING) {
					indent();
					generate(np);
					emit(false, "[strlen(");
					generate(np);
					emit(false, ")]");
				} else {
					emit(true, "_cb2crt_dummybuf[78]");
				}
				emit(false, " = '\\0';\n");
				if (np->n_val_type == VTYPE_NUM) {
					indent();
					generate(np);
					emit(false,
					    " = atol(_cb2crt_dummybuf);\n");
				}
			}
		} else {
			emit(true, "putchar('?');\n");
			emit(true, "putchar(' ');\n");
			emit(true, "while (cbm_k_getin() != '\\r')\n");
			indent_level++;
			emit(true, ";\n");
			indent_level--;
			emit(true, "putchar('\\n');\n");
		}
		break;
	}
	case TOK_INPUT_H:
		break;
	case TOK_INT:
		emit(false, "(int32_t)(");
		generate(n->n_left);
		emit(false, ")");
		break;
	case TOK_INTEGER:
		emit(false, "(int32_t)%d", n->n_num);
		break;
	case TOK_LEFT_D:
		emit(false, "_CB2CRT_LEFT_D(");
		generate(n->n_left);
		emit(false, ", _cb2crt_dummybuf, ");
		generate(n->n_mid);
		emit(false, ")");
		break;
	case TOK_LEN:
		emit(false, "strlen(");
		generate(n->n_left);
		emit(false, ")");
		break;
	case TOK_LINENO:
		while (IF_count > 0) {
			indent_level--;
			emit(true, "}\n");
			IF_count--;
		}
		emit(false, "/* line %d */\n", n->n_num);
		if (line_targets[n->n_num])
			emit(false, "line_label%d:;\n", n->n_num);
		break;
	case TOK_LIST:
		/* LIST cannot be implemented in any meaningful way */
		emit(true, "/* skipped LIST statement: %s */\n", n->n_str);
		break;
	case TOK_LOAD:
		emit(true, "clr();\n");
		/* FALLTHROUGH */
	case TOK_SAVE:
	case TOK_VERIFY:
	{
		node_t *np;
		emit(true, "cbm_k_setlfs(1, ");
		if (n->n_left != NULL && n->n_left->n_next != NULL) {
			np = n->n_left->n_next;
			generate(np);
			emit(false, ", ");
			np = np->n_next;
			if (np != NULL)
				generate(np);
			else
				emit(false, "1");
		} else {
			emit(false, "1, 1");
		}
		emit(false, ");\n");
		emit(true, "cbm_k_setnam(");
		if (n->n_left != NULL)
			generate(n->n_left);
		else
			emit(false, "\"*\"");
		emit(false, ");\n");
		emit(true, "cbm_k_open();\n");
		if (n->n_tok_type == TOK_LOAD) {
			emit(true, "cbm_k_load(0, 0);\n");
		} else if (n->n_tok_type == TOK_VERIFY) {
			emit(true, "cbm_k_load(1, 0);\n");
		} else {
			emit(true,
			    "cbm_k_save((void *)0x801, &__bss_end - 1);\n");
		}
		emit(true, "cbm_k_close(1);\n");
		break;
	}
	case TOK_LOG:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "logf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped LOG statement */");
#endif
		break;
	case TOK_LT:
	{
		enum val_type l_vtype;
		enum val_type m_vtype;
		if (n->n_num == TOK_LT) {
			/* special case for =<>, =><, <>=, and ><= */
			emit(false, "(-1L)");
			break;
		}
		l_vtype = n->n_left->n_val_type;
		m_vtype = n->n_mid->n_val_type;
		emit(false, "(");
		if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING)
			emit(false, "strcmp(");
		else
			emit(false, "(");
		generate(n->n_left);
		if (l_vtype == VTYPE_STRING && m_vtype != VTYPE_STRING)
			emit(false, "[0]");
		if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING) {
			emit(false, ", ");
		} else {
			if (n->n_num == 0)
				emit(false, ") < (");
			else if (n->n_num == TOK_EQ)
				emit(false, ") <= (");
			else if (n->n_num == TOK_GT)
				emit(false, ") != (");
		}
		generate(n->n_mid);
		if (l_vtype != VTYPE_STRING && m_vtype == VTYPE_STRING)
			emit(false, "[0]");
		if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING) {
			if (n->n_num == 0)
				emit(false, ") < 0");
			else if (n->n_num == TOK_EQ)
				emit(false, ") <= 0");
			else if (n->n_num == TOK_GT)
				emit(false, ") != 0");
		} else {
			emit(false, ")");
		}
		emit(false, " ? -1L : 0L)");
		break;
	}
	case TOK_MID_D:
		emit(false, "_CB2CRT_MID_D(");
		generate(n->n_left);
		emit(false, ", _cb2crt_dummybuf, ");
		generate(n->n_mid);
		emit(false, ", ");
		if (n->n_right != NULL) {
			generate(n->n_right);
		} else {
			emit(false, "strlen(");
			generate(n->n_left);
			emit(false, ") - (");
			generate(n->n_mid);
			emit(false, ")");
		}
		emit(false, ")");
		break;
	case TOK_MINUS:
		if (n->n_left != NULL) {
			emit(false, "(");
			generate(n->n_left);
			emit(false, ") - (");
		} else {
			emit(false, "(-(");
		}
		generate(n->n_mid);
		emit(false, ")");
		if (n->n_left == NULL)
			emit(false, ")");
		break;
	case TOK_NEW:
		/* treat NEW like END */
		emit(true, "exit(EXIT_SUCCESS);\n");
		break;
	case TOK_NEXT:
		if (n->n_num != 0) {
			/*
			 * FIXME: 1 in n_num means this is immediately
			 *        following a FOR and a delay was generated
			 *        already. This could be done better.
			 */
			break;
		}
		if (n->n_left == NULL) {
			indent_level--;
			emit(true, "}\n");
			FOR_count--;
		} else {
			node_t *np;
			for (np = n->n_left; np != NULL; np = np->n_next) {
				indent_level--;
				emit(true, "}\n");
				FOR_count--;
			}
		}
		break;
	case TOK_NOT:
		if (n->n_val_type == VTYPE_BOOL)
			emit(false, "!");
		else if (n->n_val_type == VTYPE_NUM)
			emit(false, "~(int32_t)(");
		generate(n->n_left);
		if (n->n_val_type == VTYPE_BOOL)
			emit(false, " ? -1L : 0L");
		else if (n->n_val_type == VTYPE_NUM)
			emit(false, ")");
		break;
	case TOK_ON:
	{
		node_t *np;
		int _val = 0;
		bool need_return_label = false;
		emit(true, "switch ((int32_t)(");
		generate(n->n_left);
		emit(false, ")) {\n");
		for (np = n->n_mid; np != NULL; np = np->n_next) {
			emit(true, "case %d: ", ++_val);
			if (n->n_num == TOK_GOSUB && LINE_EXISTS(np->n_num)) {
				node_t *return_stmt;
				return_stmt =
				    ptree_find_token(line_ary[np->n_num],
					TOK_RETURN);
				if (return_stmt != NULL) {
					/* the line to return to */
					emit(false, "__RETURN%d_target = "
					    "&&return_label%d; ",
					    return_stmt->n_lineno,
					    return_lbl_index + 1);
					need_return_label = true;
				}
			}
			if (!LINE_EXISTS(np->n_num)) {
				emit(false,
				    "/* no such line: %s %d */ break;\n",
				    n->n_num == TOK_GOSUB ? "gosub" : "goto",
				    np->n_num);
			} else {
				emit(false,
				    "goto line_label%d;\n", np->n_num);
			}
		}
		emit(true, "default: break;\n");
		emit(true, "}\n");
		if (need_return_label)
			emit(false, "return_label%d:;\n", ++return_lbl_index);
		break;
	}
	case TOK_OPEN:
	{
		node_t *np;
		emit(true, "cbm_k_setlfs(");
		generate(n->n_left);
		emit(false, ", ");
		np = n->n_left->n_next;
		if (np != NULL) {
			generate(np);
			emit(false, ", ");
			np = np->n_next;
			if (np != NULL)
				generate(np);
			else
				emit(false, "1");
		} else {
			emit(false, "1, 1");
		}
		emit(false, ");\n");
		if (np != NULL) {
			np = np->n_next;
			if (np != NULL) {
				if (np->n_val_type == VTYPE_CHAR) {
					emit(true, "{\n");
					indent_level++;
					emit(true, "char nam[2];\n");
					emit(true, "nam[0] = ");
					generate(np);
					emit(false, ";\n");
					emit(true, "nam[1] = '\\0';\n");
					emit(true, "cbm_k_setnam(nam);\n");
					indent_level--;
					emit(true, "}\n");
				} else {
					emit(true, "cbm_k_setnam(");
					generate(np);
					emit(false,");\n");
				}
			}
		}
		emit(true, "cbm_k_open();\n");
		break;
	}
	case TOK_OR:
		if (n->n_val_type == VTYPE_NUM)
			emit(false, "(int32_t)(");
		generate(n->n_left);
		if (n->n_val_type == VTYPE_BOOL)
			emit(false, " || ");
		else if (n->n_val_type == VTYPE_NUM)
			emit(false, ") | (int32_t)(");
		generate(n->n_mid);
		if (n->n_val_type == VTYPE_BOOL)
			emit(false, " ? -1L : 0L");
		else if (n->n_val_type == VTYPE_NUM)
			emit(false, ")");
		break;
 	case TOK_PEEK:
		emit(false, "(*(volatile char *)(uintptr_t)(");
		generate(n->n_left);
		emit(false, "))");
		break;
	case TOK_PI:
		emit(false, "(int32_t)%s", n->n_str);
		break;
	case TOK_PLUS:
	{
		enum val_type l_vtype, m_vtype;
		m_vtype = n->n_mid->n_val_type;
		if (n->n_left == NULL) {
			if (m_vtype != VTYPE_CHAR && m_vtype != VTYPE_STRING) {
				emit(false, "(");
				generate(n->n_mid);
				emit(false, ")");
			} else {
				emit(false, "_cb2crt_cwp(");
				if (m_vtype == VTYPE_CHAR) {
					generate(n->n_mid);
					emit(false,
					    ", 0, \"\", NULL, false, true)");
				} else if (m_vtype == VTYPE_STRING) {
					emit(false, "0, 0, \"\", ");
					generate(n->n_mid);
					emit(false, ", false, false)");
				}
			}
			break;
		}
		l_vtype = n->n_left->n_val_type;
		if (l_vtype != VTYPE_CHAR && l_vtype != VTYPE_STRING &&
		    m_vtype != VTYPE_CHAR && m_vtype != VTYPE_STRING) {
			emit(false, "(");
			generate(n->n_left);
			emit(false, ") + (");
			generate(n->n_mid);
			emit(false, ")");
			break;
		}
		emit(false, "_cb2crt_cwp(");
		if (l_vtype == VTYPE_CHAR && m_vtype == VTYPE_CHAR) {
			generate(n->n_left);
			emit(false, ", ");
			generate(n->n_mid);
			emit(false, ", NULL, NULL, true, true)");
		} else if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_CHAR) {
			generate(n->n_mid);
			emit(false, ", 0, ");
			generate(n->n_left);
			emit(false, ", NULL, false, true)");
		} else if (l_vtype == VTYPE_CHAR && m_vtype == VTYPE_STRING) {
			generate(n->n_left);
			emit(false, ", 0, ");
			generate(n->n_mid);
			emit(false, ", NULL, true, false)");
		} else if (l_vtype == VTYPE_STRING && m_vtype == VTYPE_STRING) {
			emit(false, "0, 0, ");
			generate(n->n_left);
			emit(false, ", ");
			generate(n->n_mid);
			emit(false, ", false, false)");
}
		break;
	}
	case TOK_POKE:
		emit(true, "*(volatile char *)(uintptr_t)(");
		generate(n->n_left);
		emit(false, ") = (char)(");
		generate(n->n_mid);
		emit(false, ");\n");
		break;
	case TOK_POS:
		emit(false, "(int32_t)_CB2CRT_PNTR");
		break;
	case TOK_PRINT:
	{
		node_t *np;
		bool semi_last = false;
		for (np = n->n_left; np != NULL; np = np->n_next) {
			semi_last = false;
			if (np->n_tok_type == TOK_ASC ||
			    np->n_tok_type == TOK_CHR_D) {
				emit(true, "putchar(");
				generate(np);
				emit(false, ");\n");
			} else if (np->n_tok_type == TOK_COMMA) {
				emit(true, "_cb2crt_print_comma();\n");
			} else if (np->n_tok_type == TOK_ID) {
				if (np->n_val_type == VTYPE_CHAR ||
				    np->n_val_type == VTYPE_STRING) {
					emit(true, "fputs(");
				} else {
					if (rflag) {
						emit(true,
						    "printf(\" %%g \", ");
					} else {
						emit(true,
						    "_cb2crt_print_signed_int(");
					}
				}
				generate(np);
				if (np->n_val_type == VTYPE_CHAR ||
				    np->n_val_type == VTYPE_STRING) {
					emit(false, ", stdout");
				}
				emit(false, ");\n");
			} else if (np->n_tok_type == TOK_LEFT_D ||
				np->n_tok_type == TOK_MID_D ||
				np->n_tok_type == TOK_RIGHT_D) {
				emit(true, "fputs(");
				generate(np);
				emit(false, ", stdout);\n");
			} else if (np->n_tok_type == TOK_LEN) {
				emit(true, "_cb2crt_print_signed_int(");
				generate(np);
				emit(false, ");\n");
			} else if (np->n_tok_type == TOK_PEEK) {
				emit(true, "_cb2crt_print_signed_int(");
				generate(np);
				emit(false, ");\n");
			} else if (np->n_tok_type == TOK_PLUS) {
				if (np->n_val_type == VTYPE_CHAR ||
				    np->n_val_type == VTYPE_STRING) {
					emit(true, "fputs(");
				} else {
					if (rflag) {
						emit(true,
						    "printf(\" %%g \", ");
					} else {
						emit(true,
						    "_cb2crt_print_signed_int(");
					}
				}
				generate(np);
				if (np->n_val_type == VTYPE_CHAR ||
				    np->n_val_type == VTYPE_STRING) {
					emit(false, ", stdout");
				}
				emit(false, ");\n");
			} else if (np->n_tok_type == TOK_SEMICOLON) {
				semi_last = true;
			} else if (np->n_tok_type == TOK_SPC) {
				emit(true, "_cb2crt_spc(");
				generate(np->n_left);
				emit(false, ");\n");
			} else if (np->n_tok_type == TOK_STR_D) {
				emit(true, "fputs(");
				generate(np);
				emit(false, ", stdout);\n");
			} else if (np->n_tok_type == TOK_STRING) {
				emit(true, "fputs(");
				output_str(np->n_str);
				emit(false, ", stdout);\n");
			} else if (np->n_tok_type == TOK_TAB) {
				emit(true, "_cb2crt_tab(");
				generate(np->n_left);
				emit(false, ");\n");
			} else {
				if (rflag) {
					emit(true, "printf(\" %%g \", ");
				} else {
					emit(true,
					    "_cb2crt_print_signed_int(");
				}
				generate(np);
				emit(false, ");\n");
			}
		}
		if (!semi_last)
			emit(true, "putchar('\\n');\n");
		break;
	}
	case TOK_PRINT_H:
		break;
	case TOK_READ:
	{
		node_t *np;
		for (np = n->n_left; np != NULL; np = np->n_next) {
			if (np->n_val_type == VTYPE_STRING)
				emit(true, "strcpy(");
			else
				indent();
			generate(np);
			if (np->n_val_type == VTYPE_STRING)
				emit(false, ", ");
			else
				emit(false, " = atol(");
			emit(false,
			    "_cb2crt_data_values[_cb2crt_data_index++]);\n");
		}
		break;
	}
	case TOK_REAL:
		emit(false, "(int32_t)%s", n->n_str);
		break;
	case TOK_REM:
		emit(true, "//'%s'\n", n->n_str);
		break;
	case TOK_RESTORE:
		if (SEEN_TOK(TOK_DATA))
			emit(true, "_cb2crt_data_index = 0;\n");
		break;
	case TOK_RETURN:
		emit(true, "goto *__RETURN%d_target;\n", n->n_lineno);
		break;
	case TOK_RIGHT_D:
		emit(false, "_CB2CRT_RIGHT_D(");
		generate(n->n_left);
		emit(false, ", _cb2crt_dummybuf, ");
		generate(n->n_mid);
		emit(false, ")");
		break;
	case TOK_RND:
		emit(false, "(rand() <= 16383 ? 0L : 1L)");
		break;
	case TOK_RUN:
		emit(true, "clr();\n");
		if (n->n_num != -1 && !LINE_EXISTS(n->n_num)) {
			emit(true, ";\n");
		} else {
			emit(true, "goto line_label%d;\n",
			    n->n_num == -1 ? first_line_number : n->n_num);
		}
		break;
	case TOK_SGN:
		emit(false, "(int32_t)(((");
		generate(n->n_left);
		emit(false, ") > 0) - ((");
		generate(n->n_left);
		emit(false, ") < 0))");
		break;
	case TOK_SIN:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "sinf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped SIN statement */");
#endif
		break;
	case TOK_SLASH:
		emit(false, "(");
		generate(n->n_left);
		emit(false, ") / (");
		generate(n->n_mid);
		emit(false, ")");
		break;
	case TOK_SPC:
		/* handled in PRINT */
		break;
	case TOK_SQR:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "sqrtf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped SQR statement */");
#endif
		break;
	case TOK_STOP:
		/* treat STOP like END */
		emit(true, "exit(EXIT_SUCCESS);\n");
		break;
	case TOK_STRING:
	{
		output_str(n->n_str);
		break;
	}
	case TOK_STR_D:
		emit(false, "_cb2crt_str_d(");
		generate(n->n_left);
		emit(false, ")");
		break;
	case TOK_SYS:
		emit(true, "_cb2crt_sys(");
		generate(n->n_left);
		emit(false, ");\n");
		break;
	case TOK_TAB:
		/* handled in PRINT */
		break;
	case TOK_TAN:
#if 0
		emit(false, "(");
		if (!rflag)
			emit(false, "(int32_t)");
		emit(false, "tanf((float)");
		generate(n->n_left);
		emit(false, "))");
#else
		emit(false, "((int32_t)1) /* skipped TAN statement */");
#endif
		break;
	case TOK_USR:
		emit(false, "_cb2crt_usr(");
		generate(n->n_left);
		emit(false, ")");
		break;
	case TOK_VAL:
		if (n->n_left->n_val_type == VTYPE_STRING) {
			if (rflag)
				emit(false, "atof");
			else
				emit(false, "atol");
		}
		emit(false, "(");
		generate(n->n_left);
		emit(false, ")");
		break;
	case TOK_WAIT:
		emit(true, "for (; ((((*(volatile char *)(uintptr_t)(");
		generate(n->n_left);
		emit(false, ")) ^ (char)(");
		if (n->n_right == NULL)
			emit(false, "0");
		else
			generate(n->n_right);
		emit(false, ")) & (char)(");
		generate(n->n_mid);
		emit(false, ")) == 0););\n");
		break;
	default:
		break;
	}
}

static void
traverse_ptree(node_t *n)
{
	if (n == NULL)
		return;
	if (dflag) {
		DEBUG_NODE(n);
		traverse_ptree(n->n_left);
		traverse_ptree(n->n_mid);
		traverse_ptree(n->n_right);
		traverse_ptree(n->n_next);
	} else {
		generate(n);
		if (n->n_tok_type == TOK_LINENO)
			traverse_ptree(n->n_left);	/* first stmt */
		traverse_ptree(n->n_next);		/* next stmt or line */
	}
}

int
gencode(node_t *n)
{
	if (!dflag) {
		global();
		main1();
	}
	traverse_ptree(n);
	if (!dflag)
		main2();
	return 0;
}
