/*
 * Copyright (c) 2024 Sascha Wildner <swildner@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#include <stdarg.h>
#include <stdbool.h>
#include <stdint.h>

#ifndef __BSD_VISIBLE
#define	__DECONST(type, var)	((type)(uintptr_t)(const void *)(var))
#define	NELEM(ary)		(sizeof(ary) / sizeof((ary)[0]))
#define	__printflike(fmtarg, firstvararg)	\
	    __attribute__((__nonnull__(fmtarg), \
			  __format__ (__printf__, fmtarg, firstvararg)))
#endif

#define	CBM_PI		"3.14159265"
#define	SYMTBL_SIZE	1024
#define	ARRAY_DIMNS	37		/* how deep we can DIM in 80 chars */
#define	BASIC_LINES	64000		/* 0 - 63999 are valid */
#define	DATA_VALS_SIZE	65536

#define	SCTYPE_U8_MIN	0
#define	SCTYPE_U8_MAX	255
#define	SCTYPE_S8_MIN	-128
#define	SCTYPE_S8_MAX	127

#define	SCTYPE_U16_MIN	0
#define	SCTYPE_U16_MAX	65535
#define	SCTYPE_S16_MIN	-32768
#define	SCTYPE_S16_MAX	32767

#define	SCTYPE_U32_MIN	0
#define	SCTYPE_U32_MAX	4294967295

#define	DEBUG_NODE(n) do {						\
	if ((n)->n_tok_type == TOK_LINENO) {				\
		printf("%lu [label = \"%d (%d)\"]\n",			\
		    (uint64_t)(n), (n)->n_tok_type, (n)->n_num);	\
	} else {							\
		printf("%lu [label = \"%d\"]\n",			\
		    (uint64_t)(n), (n)->n_tok_type);			\
	}								\
	if ((n)->n_next != NULL) {					\
		printf("%lu -> %lu [label = \"n_next\"]\n",		\
		    (uint64_t)(n), (uint64_t)(n)->n_next);		\
	}								\
	if ((n)->n_left != NULL) {					\
		printf("%lu -> %lu [label = \"n_left\"]\n",		\
		    (uint64_t)(n), (uint64_t)(n)->n_left);		\
	}								\
	if ((n)->n_mid != NULL) {					\
		printf("%lu -> %lu [label = \"n_mid\"]\n",		\
		    (uint64_t)(n), (uint64_t)(n)->n_mid);		\
	}								\
	if ((n)->n_right != NULL) {					\
		printf("%lu -> %lu [label = \"n_right\"]\n",		\
		    (uint64_t)(n), (uint64_t)(n)->n_right);		\
	}								\
} while (0)

#define	LINE_EXISTS(l)	((uint64_t)(l) < BASIC_LINES && line_ary[(l)] != NULL)
#define	SEEN_TOK(t)	(tok_count[(t)] > 0)

#define	SEEN_LMR	(SEEN_TOK(TOK_LEFT_D) || SEEN_TOK(TOK_MID_D) ||	\
			 SEEN_TOK(TOK_RIGHT_D))
#define	SEEN_MATH	(SEEN_TOK(TOK_ATN) || SEEN_TOK(TOK_COS) ||	\
			 SEEN_TOK(TOK_EXP) || SEEN_TOK(TOK_LOG) ||	\
			 SEEN_TOK(TOK_SIN) || SEEN_TOK(TOK_SQR) ||	\
			 SEEN_TOK(TOK_TAN))

enum diag_type {
	DIAG_WARN = 1,
	DIAG_ERR
};

enum sym_btype {
	SBTYPE_NONE = 1,
	SBTYPE_DEF,
	SBTYPE_FN,
	SBTYPE_INT,
	SBTYPE_REAL,
	SBTYPE_RETURN,
	SBTYPE_STRING
};

enum sym_ctype_num {
	SCTYPE_NONE = 1,
	SCTYPE_U8,
	SCTYPE_S8,
	SCTYPE_U16,
	SCTYPE_S16,
	SCTYPE_U32,
	SCTYPE_S32
};

enum val_type {
	VTYPE_NONE = 1,
	VTYPE_BOOL,
	VTYPE_CHAR,
	VTYPE_NUM,
	VTYPE_STRING
};

typedef struct _node {
	int n_tok_type;
	enum val_type n_val_type;
	unsigned n_lineno;
	int n_num;
	char *n_str;
	struct _node *n_left;
	struct _node *n_mid;
	struct _node *n_right;
	struct _node *n_next;
} node_t;

typedef struct _symbol {
	const char *s_name;
	const char *s_fnarg;
	enum sym_btype s_btype;
	bool s_auto_ctype;
	long long sym_ctype_lsc, sym_ctype_hsc;
	enum sym_ctype_num s_ctype_num;
	size_t s_ctype_strlen;
	int s_ctype_dimns[ARRAY_DIMNS];
	node_t *s_np;
	struct _symbol *s_next;
} symbol_t;

typedef struct {
	symbol_t **st_table;
	size_t st_size;
	size_t st_capacity;
} symbol_table_t;

/* basic20_parse.y */
#ifdef DEBUG
extern int		yydebug;
#endif
extern node_t		*line_ary[], *top_node;
extern symbol_table_t	*symtbl;
extern int		first_line_number;
extern bool		line_targets[];
node_t			*ptree_find_token(node_t *, int);
void			ptree_free_tree(node_t *);
void			yyerror(const char *);

/* basic20_scan.l */
extern int		data_index, line_number, col_number, prev_line;
extern int		prev_col, tok_count[];
extern unsigned		basic_lineno;
extern const char 	*data_values[], *line_buf;
void			STRING_convert(const char *, char *, size_t);
void			free_data_values(void);
void			toeol(bool);
#ifndef __BSD_VISIBLE
int			yylex(void);
#endif

/* cb2c.c */
extern bool		dflag, rflag;
void			cb2c_diag(enum diag_type, const char *, ...)
			    __printflike(2, 3);

/* codegen.c */
int			gencode(node_t *);

/* symtbl.c */
void			adjust_sym_ctype(node_t *);
symbol_table_t		*create_symtbl(size_t);
void			free_symtbl(symbol_table_t *);
symbol_t		*insert_sym(symbol_table_t *, const char *,
			    const char *, enum sym_btype, bool, node_t *);
symbol_t		*lookup_sym(symbol_table_t *, const char *);
