/*
 * Copyright (c) 2024 Sascha Wildner <swildner@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "cb2c.h"

extern int	yyparse(void);

static void	cb2c_cleanup(void);
static void	cb2c_usage(void);

static const char *infile, *outfile;
static FILE *infp, *outfp;
#ifdef DEBUG
static const char *optstr = "dho:vV";
#else
static const char *optstr = "dho:V";
#endif
static bool	hflag;

bool		dflag, rflag;
#ifdef DEBUG
bool		vflag;
#endif

static void
cb2c_cleanup(void)
{
	if (infp)
		fclose(infp);
	if (outfp)
		fclose(outfp);
	free_data_values();
	free_symtbl(symtbl);
	ptree_free_tree(top_node);
}

/* this function is for scanning/parsing, not for code generation */
void
cb2c_diag(enum diag_type type, const char *fmt, ...)
{
	va_list ap;
	int i;

	/* diag msg */
	va_start(ap, fmt);
	fprintf(stderr, "%s:%d:%d: %s: ",
	    infile, line_number, prev_col + 1,
	    type == DIAG_WARN ? "warning" : "error");
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	/* src line */
	fprintf(stderr, "%s\n", line_buf);
	/* caret indicator */
	for (i = 0; i < prev_col; i++)
		fprintf(stderr, " ");
	fprintf(stderr, "^");
	for (i = 0; i < col_number - prev_col - 1; i++)
		fprintf(stderr, "~");
	fprintf(stderr, "\n");
	if (type == DIAG_ERR)
		exit(EXIT_FAILURE);
}

static void
cb2c_usage(void)
{
#ifdef DEBUG
	fprintf(stderr, "usage: cb2c [-dhvV] [-o outfile] [infile]\n");
#else
	fprintf(stderr, "usage: cb2c [-dhV] [-o outfile] [infile]\n");
#endif
	if (hflag) {
		fprintf(stderr, "  -d            "
		    "print a graphviz representation of the parse tree\n");
		fprintf(stderr, "  -h            "
		    "print this help\n");
		fprintf(stderr, "  -o outfile    "
		    "specify the output file name\n");
#ifdef DEBUG
		fprintf(stderr, "  -v            "
		    "turn on lex/yacc debugging\n");
#endif
		fprintf(stderr, "  -V            "
		    "print version information\n");
	}
	exit(EXIT_FAILURE);
}

int
main(int argc, char *argv[])
{
	char ch;
	int rc;

	while ((ch = getopt(argc, argv, optstr)) != -1) {
		switch (ch) {
		case 'd':
			dflag = true;
			break;
		case 'o':
			outfile = optarg;
			break;
#ifdef DEBUG
		case 'v':
			vflag = true;
			break;
#endif
		case 'V':
			printf("%s\n", CB2C_VERSION);
			return 0;
			break;
		case 'h':
			hflag = true;
			/* FALLTHROUGH */
		default:
			cb2c_usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;
	if (argc > 1)
		cb2c_usage();
	if (argv[0] && (argv[0][0] != '-' || argv[0][1] != '\0'))
		infile = argv[0];
	if (infile) {
		if ((infp = freopen(infile, "r", stdin)) == NULL)
			err(1, "%s", infile);
	} else {
		infile = "<stdin>";
	}
	if (outfile == NULL)
		outfile = "c.out";
#ifdef DEBUG
	if (vflag)
		yydebug = 1;
#endif
	line_number = 1;
	symtbl = create_symtbl(SYMTBL_SIZE);
	rc = yyparse();
	if (rc == 0) {
		if (dflag) {
			printf("digraph {\n");
		} else if (outfile[0] != '-' || outfile[1] != '\0') {
			if ((outfp = freopen(outfile, "w", stdout)) == NULL)
				err(1, "%s", outfile);
		} else {
			outfile = "<stdout>";
		}
		rc = gencode(top_node);
		if (dflag)
			printf("}\n");
	}
	cb2c_cleanup();
	return rc;
}
