/*
 * Copyright (c) 2024 Sascha Wildner <swildner@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cb2c.h"

#include "basic20_parse.h"

static unsigned
djb2(const char *str, size_t capacity)
{
	unsigned hash = 5381;
	int c;

	while ((c = *str++))
		hash = ((hash << 5) + hash) + c;
	return hash % capacity;
}

symbol_table_t *
create_symtbl(size_t capacity)
{
	symbol_table_t *st =
	    (symbol_table_t *)calloc(1, sizeof(symbol_table_t));

	st->st_table = (symbol_t **)calloc(capacity, sizeof(symbol_t *));
	st->st_size = 0;
	st->st_capacity = capacity;
	return st;
}

symbol_t *
insert_sym(symbol_table_t *st, const char *name, const char *fnarg,
    enum sym_btype btype, bool auto_ctype, node_t *np)
{
	unsigned index;
	symbol_t *s;

	if ((s = lookup_sym(st, name)) == NULL) {
		index = djb2(name, st->st_capacity);
		s = (symbol_t *)calloc(1, sizeof(symbol_t));
		s->s_name = strdup(name);
		if (fnarg != NULL)
			s->s_fnarg = strdup(fnarg);
		s->s_btype = btype;
		s->s_auto_ctype = auto_ctype;
		s->s_ctype_num = SCTYPE_NONE;
		s->s_np = np;
		s->s_next = st->st_table[index];
		st->st_table[index] = s;
		st->st_size++;
	}
	return s;
}

symbol_t *
lookup_sym(symbol_table_t *st, const char *name)
{
	unsigned int index = djb2(name, st->st_capacity);
	symbol_t *symbol = st->st_table[index];

	while (symbol != NULL && strcmp(symbol->s_name, name) != 0)
		symbol = symbol->s_next;
	return symbol;
}

static void
adjust_sym_ctype_num(symbol_t *sp, int value)
{
	sp->s_ctype_num = SCTYPE_S32;
	if (value < sp->sym_ctype_lsc)
		sp->sym_ctype_lsc = value;
	if (value > sp->sym_ctype_hsc)
		sp->sym_ctype_hsc = value;
	if (sp->sym_ctype_lsc >= SCTYPE_U32_MIN)
		sp->s_ctype_num = SCTYPE_U32;
	if (sp->sym_ctype_lsc >= SCTYPE_S16_MIN &&
	    sp->sym_ctype_hsc <= SCTYPE_S16_MAX) {
		sp->s_ctype_num = SCTYPE_S16;
	}
	if (sp->sym_ctype_lsc >= SCTYPE_U16_MIN &&
	    sp->sym_ctype_hsc <= SCTYPE_U16_MAX) {
		sp->s_ctype_num = SCTYPE_U16;
	}
	if (sp->sym_ctype_lsc >= SCTYPE_S8_MIN &&
	    sp->sym_ctype_hsc <= SCTYPE_S8_MAX) {
		sp->s_ctype_num = SCTYPE_S8;
	}
	if (sp->sym_ctype_lsc >= SCTYPE_U8_MIN &&
	    sp->sym_ctype_hsc <= SCTYPE_U8_MAX) {
		sp->s_ctype_num = SCTYPE_U8;
	}
}

void
adjust_sym_ctype(node_t *np)
{
	symbol_t *sp;

	switch (np->n_tok_type) {
	case TOK_EQ:
		if (np->n_mid == NULL ||
		    (sp = lookup_sym(symtbl, np->n_left->n_str)) == NULL ||
		    !sp->s_auto_ctype) {
			return;
		}
		switch (np->n_mid->n_tok_type) {
		case TOK_MINUS:
			if (np->n_mid->n_left == NULL &&
			    np->n_mid->n_mid->n_tok_type == TOK_INTEGER) {
				adjust_sym_ctype_num(sp, -1 * np->n_mid->n_mid->n_num);
			} else {
				sp->s_auto_ctype = false;
			}
			break;
		case TOK_PLUS:
			if (np->n_mid->n_left == NULL &&
			    np->n_mid->n_mid->n_tok_type == TOK_INTEGER) {
				adjust_sym_ctype_num(sp, np->n_mid->n_mid->n_num);
			} else {
				sp->s_auto_ctype = false;
			}
			break;
		case TOK_INTEGER:
			adjust_sym_ctype_num(sp, np->n_mid->n_num);
			break;
		case TOK_STRING:
		{
			size_t len = strlen(np->n_mid->n_str) + 1;
			if (len > sp->s_ctype_strlen)
				sp->s_ctype_strlen = len;
			break;
		}
		case TOK_ASC:
		case TOK_LEN:
		case TOK_PEEK:
		case TOK_PI:
			adjust_sym_ctype_num(sp, SCTYPE_U8_MAX);
			break;
		case TOK_SGN:
			adjust_sym_ctype_num(sp, SCTYPE_S8_MIN);
			break;
		case TOK_FRE:
			adjust_sym_ctype_num(sp, SCTYPE_S16_MAX);
			break;
		default:
			sp->s_auto_ctype = false;
			break;
		}
		break;
	case TOK_FOR:
		if ((sp = lookup_sym(symtbl, np->n_str)) == NULL ||
		    !sp->s_auto_ctype) {
			return;
		}
		if (np->n_left->n_tok_type == TOK_INTEGER &&
		    np->n_mid->n_tok_type == TOK_INTEGER) {
			if (np->n_right == NULL) {
				adjust_sym_ctype_num(sp, np->n_left->n_num);
				adjust_sym_ctype_num(sp, np->n_mid->n_num);
			} else {
				adjust_sym_ctype_num(sp,
				    np->n_left->n_num - np->n_right->n_num);
				adjust_sym_ctype_num(sp,
				    np->n_left->n_num + np->n_right->n_num);
				adjust_sym_ctype_num(sp,
				    np->n_mid->n_num - np->n_right->n_num);
				adjust_sym_ctype_num(sp,
				    np->n_mid->n_num + np->n_right->n_num);
			}
		} else {
				sp->s_auto_ctype = false;
		}
		break;
	case TOK_GET:
	{
		node_t *np2;
		for (np2 = np->n_left; np2 != NULL; np2 = np2->n_next) {
			sp = lookup_sym(symtbl, np2->n_str);
			if (sp != NULL) {
				if (sp->s_btype == SBTYPE_INT) {
					sp->s_ctype_num = SCTYPE_U8;
				} else if (sp->s_btype == SBTYPE_STRING) {
					if (sp->s_ctype_strlen < 2)
						sp->s_ctype_strlen = 2;
				} else {
					sp->s_auto_ctype = false;
				}
			}
		}
		break;
	}
	case TOK_INPUT:
	case TOK_INPUT_H:
	case TOK_READ:
	{
		node_t *np2;
		for (np2 = np->n_left; np2 != NULL; np2 = np2->n_next) {
			sp = lookup_sym(symtbl, np2->n_str);
			if (sp != NULL)
				sp->s_auto_ctype = false;
		}
		break;
	}
	default:
		break;
	}
}

void
free_symtbl(symbol_table_t *st)
{
	for (size_t i = 0; i < st->st_capacity; i++) {
		symbol_t *symbol = st->st_table[i];
		while (symbol != NULL) {
			symbol_t *temp = symbol;
			symbol = symbol->s_next;
			free(__DECONST(void *, temp->s_name));
			if (temp->s_fnarg != NULL)
				free(__DECONST(void *, temp->s_fnarg));
			free(temp);
		}
	}
	free(st->st_table);
	free(st);
}
