%{
/*
 * Copyright (c) 2024 Sascha Wildner <swildner@gmail.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cb2c.h"

static char	*fnarg_id, IDbuf[256];

static enum sym_btype ID_to_cvar(char *, node_t *);
static node_t	*ptree_create_node(int, int, const char *, node_t *, node_t *,
		    node_t *);

node_t		*line_ary[BASIC_LINES], *top_node;
symbol_table_t	*symtbl;
int		first_line_number = -1;
bool		line_targets[BASIC_LINES];
%}

%union {
	node_t *node;
	char *str;
	int num;
}

%token	TOK_ABS
%token	TOK_AND
%token	TOK_ARROWUP
%token	TOK_ASC
%token	TOK_ASTERISK
%token	TOK_ATN
%token	TOK_CHR_D
%token	TOK_CLOSE
%token	TOK_CLR
%token	TOK_CMD
%token	TOK_COLON
%token	TOK_COMMA
%token	TOK_CONT
%token	TOK_COS
%token	TOK_CPAREN
%token	TOK_DATA
%token	TOK_DEF
%token	TOK_DIM
%token	TOK_END
%token	TOK_EOL
%token	TOK_EQ
%token	TOK_EXP
%token	TOK_FN
%token	TOK_FOR
%token	TOK_FRE
%token	TOK_GET
%token	TOK_GO
%token	TOK_GOSUB
%token	TOK_GOTO
%token	TOK_GT
%token	TOK_HASH
%token	TOK_IF
%token	TOK_INPUT
%token	TOK_INPUT_H
%token	TOK_INT
%token	TOK_LEFT_D
%token	TOK_LEN
%token	TOK_LET
%token	TOK_LOAD
%token	TOK_LOG
%token	TOK_LT
%token	TOK_MID_D
%token	TOK_MINUS
%token	TOK_NEW
%token	TOK_NEXT
%token	TOK_NOT
%token	TOK_ON
%token	TOK_OPAREN
%token	TOK_OPEN
%token	TOK_OR
%token	TOK_PEEK
%token	TOK_PLUS
%token	TOK_POKE
%token	TOK_POS
%token	TOK_PRINT
%token	TOK_PRINT_H
%token	TOK_READ
%token	TOK_RESTORE
%token	TOK_RETURN
%token	TOK_RIGHT_D
%token	TOK_RND
%token	TOK_RUN
%token	TOK_SAVE
%token	TOK_SEMICOLON
%token	TOK_SGN
%token	TOK_SIN
%token	TOK_SLASH
%token	TOK_SPC
%token	TOK_SQR
%token	TOK_STEP
%token	TOK_STOP
%token	TOK_STR_D
%token	TOK_SYS
%token	TOK_TAB
%token	TOK_TAN
%token	TOK_THEN
%token	TOK_TO
%token	TOK_USR
%token	TOK_VAL
%token	TOK_VERIFY
%token	TOK_WAIT

%token <num>	TOK_INTEGER
%token <num>	TOK_LINENO

%token <str>	TOK_ID
%token <str>	TOK_LIST
%token <str>	TOK_PI
%token <str>	TOK_REAL
%token <str>	TOK_REM
%token <str>	TOK_STRING

%token		PREC_UNARY_PM

%type <node>	Comma
%type <node>	Constant
%type <node>	FNARG_ID
%type <node>	ID
%type <node>	IDList
%type <node>	Line
%type <node>	Lines
%type <node>	PrintList
%type <node>	PrintListElement
%type <node>	Semicolon
%type <node>	Statement
%type <node>	Statements
%type <node>	ThenClause
%type <node>	Value
%type <node>	LineNumberList
%type <node>	LineNumber
%type <node>	ExpressionList
%type <node>	Expression
%type <node>	SubExpression
%type <node>	AddExpression
%type <node>	AndExpression
%type <node>	CompareExpression
%type <node>	MultExpression
%type <node>	NotExpression
%type <node>	OrExpression
%type <node>	PowerExpression
%type <node>	SignedExpression

%type <num>	Sign

%left		TOK_OR
%left		TOK_AND
%right		TOK_NOT
%left		TOK_EQ TOK_GT TOK_LT
%left		TOK_MINUS TOK_PLUS
%left		TOK_ASTERISK TOK_SLASH
%right		PREC_UNARY_PM
%left		TOK_ARROWUP

%%

Lines:
	Lines Line {
		if ($1 != NULL) {
			node_t *last_line = $1;
			while (last_line->n_next != NULL)
				last_line = last_line->n_next;
			last_line->n_next = $2;
		}
		$$ = $1;
	}
	| Line {
		if (top_node == NULL)
			top_node = $$;
	}
	;

Line:
	TOK_LINENO Statements TOK_EOL {
		$$ = line_ary[$1] = ptree_create_node(TOK_LINENO, $1, NULL,
		    $2, NULL, NULL);
		if (first_line_number == -1)
			first_line_number = $1;
	}
	| TOK_LINENO TOK_EOL {
		$$ = line_ary[$1] = ptree_create_node(TOK_LINENO, $1, NULL,
		    NULL, NULL, NULL);
		if (first_line_number == -1)
			first_line_number = $1;
	}
	;

Statements:
	Statements TOK_COLON Statement {
		if ($1 != NULL) {
			node_t *last_stmt = $1;
			while (last_stmt->n_next != NULL)
				last_stmt = last_stmt->n_next;
			last_stmt->n_next = $3;
		}
		$$ = $1;
	}
	| Statements TOK_COLON
	| Statement
	;

Statement:
	TOK_CLOSE ExpressionList {
		$$ = ptree_create_node(TOK_CLOSE, 0, NULL, $2, NULL, NULL);
	}
	| TOK_CLR {
		$$ = ptree_create_node(TOK_CLR, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_CMD Expression {
		$$ = ptree_create_node(TOK_CMD, 0, NULL, $2, NULL, NULL);
	}
	| TOK_CMD Expression TOK_COMMA PrintList {
		$$ = ptree_create_node(TOK_CMD, 0, NULL, $2, $4, NULL);
	}
	| TOK_CONT {
		$$ = ptree_create_node(TOK_CONT, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_DATA {
		$$ = ptree_create_node(TOK_DATA, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_DEF TOK_FN TOK_ID TOK_OPAREN FNARG_ID TOK_CPAREN TOK_EQ Expression {
		char fullname[16], fullname2[16];
		sprintf(fullname, "fn%s", $3);
		$$ = ptree_create_node(TOK_DEF, 0, fullname, $8, NULL, NULL);
		free($3);
		insert_sym(symtbl, fullname, NULL, SBTYPE_FN, false, NULL);
		sprintf(fullname2, "%s_%p", fullname, $8);
		insert_sym(symtbl, fullname2, $5->n_str, SBTYPE_DEF, false, $8);
		fnarg_id = NULL;
	}
	| TOK_DIM IDList {
		node_t *np, *np2;
		symbol_t *sp;
		int i;
		$$ = ptree_create_node(TOK_DIM, 0, NULL, $2, NULL, NULL);
		/* all IDs */
		for (np = $2; np != NULL; np = np->n_next) {
			sp = lookup_sym(symtbl, np->n_str);
			/* all dimensions */
			for (np2 = np->n_left, i = 0;
			     np2 != NULL && np2->n_tok_type == TOK_INTEGER;
			     np2 = np2->n_next, i++) {
				if (sp->s_ctype_dimns[i] < np2->n_num) {
					sp->s_ctype_dimns[i] =
					    np2->n_num;
				}
			}
		}
	}
	| TOK_END {
		$$ = ptree_create_node(TOK_END, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_FOR ID TOK_EQ Expression TOK_TO Expression {
		char condname[16];
		symbol_t *sp, *sp2;
		$$ = ptree_create_node(TOK_FOR, 0, $2->n_str, $4, $6, NULL);
		sprintf(condname, "%s_cond", $2->n_str);
		sp = lookup_sym(symtbl, $2->n_str);
		free($2->n_str);
		free($2);
		if (sp != NULL) {
			sp2 = lookup_sym(symtbl, condname);
			if (sp2 == NULL) {
				sp2 = insert_sym(symtbl, condname, NULL,
				    SBTYPE_INT, true, NULL);
			}
			if (sp->s_auto_ctype)
				sp2->s_ctype_num = sp->s_ctype_num;
			else
				sp2->s_ctype_num = SCTYPE_S32;
		}
	}
	| TOK_FOR ID TOK_EQ Expression TOK_TO Expression TOK_STEP Expression {
		char condname[16];
		symbol_t *sp, *sp2;
		$$ = ptree_create_node(TOK_FOR, 0, $2->n_str, $4, $6, $8);
		sprintf(condname, "%s_cond", $2->n_str);
		sp = lookup_sym(symtbl, $2->n_str);
		free($2->n_str);
		free($2);
		if (sp != NULL) {
			sp2 = lookup_sym(symtbl, condname);
			if (sp2 == NULL) {
				sp2 = insert_sym(symtbl, condname, NULL,
				    SBTYPE_INT, true, NULL);
			}
			if (sp->s_auto_ctype)
				sp2->s_ctype_num = sp->s_ctype_num;
			else
				sp2->s_ctype_num = SCTYPE_S32;
		}
	}
	| TOK_GET IDList {
		$$ = ptree_create_node(TOK_GET, 0, NULL, $2, NULL, NULL);
	}
	| TOK_GET TOK_HASH Expression TOK_COMMA IDList {
		$$ = ptree_create_node(TOK_GET, 0, NULL, $5, $3, NULL);
	}
	| TOK_GOSUB {
		$$ = ptree_create_node(TOK_GOSUB, 0, NULL, NULL, NULL, NULL);
		line_targets[0] = true;
	}
	| TOK_GOSUB TOK_INTEGER {
		toeol(true);
		if ((uintmax_t)$2 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_GOSUB, $2, NULL, NULL, NULL, NULL);
		line_targets[$2] = true;
	}
	| TOK_GOSUB TOK_REAL {
		toeol(true);
		if ((uintmax_t)atoi($2) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_GOSUB, atoi($2), NULL, NULL, NULL, NULL);
		line_targets[atoi($2)] = true;
		free($2);
	}
	| TOK_GOTO {
		$$ = ptree_create_node(TOK_GOTO, 0, NULL, NULL, NULL, NULL);
		line_targets[0] = true;
	}
	| TOK_GOTO TOK_INTEGER {
		toeol(false);
		if ((uintmax_t)$2 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_GOTO, $2, NULL, NULL, NULL, NULL);
		line_targets[$2] = true;
	}
	| TOK_GOTO TOK_REAL {
		toeol(false);
		if ((uintmax_t)atoi($2) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_GOTO, atoi($2), NULL, NULL, NULL, NULL);
		line_targets[atoi($2)] = true;
		free($2);
	}
	| TOK_GO TOK_TO {
		$$ = ptree_create_node(TOK_GOTO, 0, NULL, NULL, NULL, NULL);
		line_targets[0] = true;
	}
	| TOK_GO TOK_TO TOK_INTEGER {
		toeol(false);
		if ((uintmax_t)$3 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_GOTO, $3, NULL, NULL, NULL, NULL);
		line_targets[$3] = true;
	}
	| TOK_GO TOK_TO TOK_REAL {
		toeol(false);
		if ((uintmax_t)atoi($3) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_GOTO, atoi($3), NULL, NULL, NULL, NULL);
		line_targets[atoi($3)] = true;
		free($3);
	}
	| TOK_IF Expression TOK_GOTO TOK_INTEGER {
		toeol(false);
		if ((uintmax_t)$4 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_IF, $4, NULL, $2, NULL, NULL);
		line_targets[$4] = true;
	}
	| TOK_IF Expression TOK_GOTO TOK_REAL {
		toeol(false);
		if ((uintmax_t)atoi($4) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_IF, atoi($4), NULL, $2, NULL, NULL);
		line_targets[atoi($4)] = true;
		free($4);
	}
	| TOK_IF Expression TOK_THEN {
		$$ = ptree_create_node(TOK_IF, -1, NULL, $2, NULL, NULL);
	}
	| TOK_IF Expression TOK_THEN ThenClause {
		$$ = ptree_create_node(TOK_IF, -1, NULL, $2, $4, NULL);
	}
	| TOK_INPUT {
		$$ = ptree_create_node(TOK_INPUT, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_INPUT IDList {
		$$ = ptree_create_node(TOK_INPUT, 0, NULL, $2, NULL, NULL);
	}
	| TOK_INPUT TOK_STRING TOK_SEMICOLON {
		$$ = ptree_create_node(TOK_INPUT, 0, $2, NULL, NULL, NULL);
		free($2);
	}
	| TOK_INPUT TOK_STRING TOK_SEMICOLON IDList {
		$$ = ptree_create_node(TOK_INPUT, 0, $2, $4, NULL, NULL);
		free($2);
	}
	| TOK_INPUT_H Expression TOK_COMMA IDList {
		$$ = ptree_create_node(TOK_INPUT_H, 0, NULL, $4, $2, NULL);
	}
	| TOK_LET ID TOK_EQ Expression {
		$$ = ptree_create_node(TOK_EQ, 0, NULL, $2, $4, NULL);
	}
	| TOK_LIST {
		$$ = ptree_create_node(TOK_LIST, 0, $1, NULL, NULL, NULL);
		free($1);
	}
	| TOK_LOAD {
		$$ = ptree_create_node(TOK_LOAD, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_LOAD ExpressionList {
		$$ = ptree_create_node(TOK_LOAD, 0, NULL, $2, NULL, NULL);
	}
	| TOK_NEW {
		$$ = ptree_create_node(TOK_NEW, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_NEXT {
		$$ = ptree_create_node(TOK_NEXT, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_NEXT IDList {
		$$ = ptree_create_node(TOK_NEXT, 0, NULL, $2, NULL, NULL);
	}
	| TOK_ON Expression TOK_GOSUB LineNumberList {
		$$ = ptree_create_node(TOK_ON, TOK_GOSUB, NULL, $2, $4, NULL);
	}
	| TOK_ON Expression TOK_GOTO LineNumberList {
		$$ = ptree_create_node(TOK_ON, TOK_GOTO, NULL, $2, $4, NULL);
	}
	| TOK_OPEN ExpressionList {
		$$ = ptree_create_node(TOK_OPEN, 0, NULL, $2, NULL, NULL);
	}
	| TOK_POKE Expression TOK_COMMA Expression {
		$$ = ptree_create_node(TOK_POKE, 0, NULL, $2, $4, NULL);
	}
	| TOK_PRINT {
		$$ = ptree_create_node(TOK_PRINT, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_PRINT PrintList {
		$$ = ptree_create_node(TOK_PRINT, 0, NULL, $2, NULL, NULL);
	}
	| TOK_PRINT_H Expression {
		$$ = ptree_create_node(TOK_PRINT_H, 0, NULL, $2, NULL, NULL);
	}
	| TOK_PRINT_H Expression TOK_COMMA {
		$$ = ptree_create_node(TOK_PRINT_H, 0, NULL, $2, NULL, NULL);
	}
	| TOK_PRINT_H Expression TOK_COMMA PrintList {
		$$ = ptree_create_node(TOK_PRINT_H, 0, NULL, $2, $4, NULL);
	}
	| TOK_READ IDList {
		$$ = ptree_create_node(TOK_READ, 0, NULL, $2, NULL, NULL);
	}
	| TOK_REM {
		$$ = ptree_create_node(TOK_REM, 0, $1, NULL, NULL, NULL);
		free($1);
	}
	| TOK_RETURN {
		char rlfvar[32];
		$$ = ptree_create_node(TOK_RETURN, 0, NULL, NULL, NULL, NULL);
		sprintf(rlfvar, "__RETURN%d_target", $$->n_lineno);
		insert_sym(symtbl, rlfvar, NULL, SBTYPE_RETURN, false, NULL);
	}
	| TOK_RESTORE {
		$$ = ptree_create_node(TOK_RESTORE, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_RUN {
		$$ = ptree_create_node(TOK_RUN, -1, NULL, NULL, NULL, NULL);
		line_targets[first_line_number] = true;
	}
	| TOK_RUN TOK_INTEGER {
		if ((uintmax_t)$2 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_RUN, $2, NULL, NULL, NULL, NULL);
		line_targets[$2] = true;
	}
	| TOK_RUN TOK_REAL {
		if ((uintmax_t)atoi($2) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_RUN, atoi($2), NULL, NULL, NULL, NULL);
		line_targets[atoi($2)] = true;
		free($2);
	}
	| TOK_SAVE ExpressionList {
		$$ = ptree_create_node(TOK_SAVE, 0, NULL, $2, NULL, NULL);
	}
	| TOK_STOP {
		$$ = ptree_create_node(TOK_STOP, 0, NULL, NULL, NULL, NULL);
	}
	| TOK_SYS ExpressionList {
		$$ = ptree_create_node(TOK_SYS, 0, NULL, $2, NULL, NULL);
	}
	| TOK_VERIFY ExpressionList {
		$$ = ptree_create_node(TOK_VERIFY, 0, NULL, $2, NULL, NULL);
	}
	| TOK_WAIT Expression TOK_COMMA Expression {
		$$ = ptree_create_node(TOK_WAIT, 0, NULL, $2, $4, NULL);
	}
	| TOK_WAIT Expression TOK_COMMA Expression TOK_COMMA Expression {
		$$ = ptree_create_node(TOK_WAIT, 0, NULL, $2, $4, $6);
	}
	| ID TOK_EQ Expression {
		$$ = ptree_create_node(TOK_EQ, 0, NULL, $1, $3, NULL);
	}
	;

FNARG_ID:
	TOK_ID TOK_OPAREN ExpressionList TOK_CPAREN {
		enum sym_btype btype = ID_to_cvar($1, $3);
		fnarg_id = IDbuf;
		$$ = ptree_create_node(TOK_ID, btype, IDbuf, $3, NULL, NULL);
		free($1);
	}
	| TOK_ID {
		enum sym_btype btype = ID_to_cvar($1, NULL);
		fnarg_id = IDbuf;
		$$ = ptree_create_node(TOK_ID, btype, IDbuf, NULL, NULL, NULL);
		free($1);
	}
	;

/* FIXME: 1 shift/reduce conflict */
ID:
	TOK_ID TOK_OPAREN ExpressionList TOK_CPAREN {
		enum sym_btype btype = ID_to_cvar($1, $3);
		$$ = ptree_create_node(TOK_ID, btype, IDbuf, $3, NULL, NULL);
		free($1);
		insert_sym(symtbl, IDbuf, NULL, btype, true, $3);
	}
	| TOK_ID {
		enum sym_btype btype = ID_to_cvar($1, NULL);
		$$ = ptree_create_node(TOK_ID, btype, IDbuf, NULL, NULL, NULL);
		free($1);
		/*
		 * Don't declare if it is a function argument or one of the
		 * special variables ST and TI. TI$ we do declare.
		 */
		if (!(fnarg_id != NULL && strcmp(IDbuf, fnarg_id) == 0) &&
		    strcmp(IDbuf, rflag ? "r_ti" : "I_ti") != 0 &&
		    strcmp(IDbuf, rflag ? "r_st" : "I_st") != 0) {
			insert_sym(symtbl, IDbuf, NULL, btype, true, NULL);
		}
	}
	;

IDList:
	IDList TOK_COMMA ID {
		if ($1 != NULL) {
			node_t *last_id = $1;
			while (last_id->n_next != NULL)
				last_id = last_id->n_next;
			last_id->n_next = $3;
		}
		$$ = $1;
	}
	| ID
	;

Comma:
	TOK_COMMA {
		$$ = ptree_create_node(TOK_COMMA, 0, NULL, NULL, NULL, NULL);
	}

Semicolon:
	TOK_SEMICOLON {
		$$ = ptree_create_node(TOK_SEMICOLON, 0, NULL,
		    NULL, NULL, NULL);
	}

PrintListElement:
	Comma
	| Expression
	| Semicolon
	;
	    
PrintList:
	PrintList PrintListElement {
		if ($1 != NULL) {
			node_t *last_expr = $1;
			while (last_expr->n_next != NULL)
				last_expr = last_expr->n_next;
			last_expr->n_next = $2;
		}
		$$ = $1;
	}
	| PrintListElement
	;

ThenClause:
	TOK_INTEGER {
		toeol(false);
		if ((uintmax_t)$1 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_INTEGER, $1, NULL, NULL, NULL, NULL);
		line_targets[$1] = true; /* without explicit GOTO */
	}
	| TOK_REAL {
		toeol(false);
		if ((uintmax_t)atoi($1) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_INTEGER, atoi($1), NULL, NULL, NULL, NULL);
		line_targets[atoi($1)] = true; /* without explicit GOTO */
		free($1);
	}
	| Statement
	;

LineNumberList:
	LineNumberList TOK_COMMA LineNumber {
		if ($1 != NULL) {
			node_t *last_int = $1;
			while (last_int->n_next != NULL)
				last_int = last_int->n_next;
			last_int->n_next = $3;
		}
		$$ = $1;
	}
	| LineNumber
	;

LineNumber:
	TOK_INTEGER {
		if ((uintmax_t)$1 >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_INTEGER, $1, NULL, NULL, NULL, NULL);
		line_targets[$1] = true;
	}
	| TOK_REAL {
		if ((uintmax_t)atoi($1) >= BASIC_LINES)
			yyerror("invalid line number");
		$$ = ptree_create_node(TOK_INTEGER, atoi($1), NULL, NULL, NULL, NULL);
		line_targets[atoi($1)] = true;
		free($1);
	}
	| {
		$$ = ptree_create_node(TOK_INTEGER, 0, NULL, NULL, NULL, NULL);
		line_targets[0] = true;
	}
	;

ExpressionList:
	ExpressionList TOK_COMMA Expression {
		if ($1 != NULL) {
			node_t *last_expr = $1;
			while (last_expr->n_next != NULL)
				last_expr = last_expr->n_next;
			last_expr->n_next = $3;
		}
		$$ = $1;
	}
	| Expression
	;

Expression:
	AndExpression
	| OrExpression
	| CompareExpression
	| AddExpression
	| MultExpression
	| PowerExpression
	| SignedExpression
	| SubExpression
	;

OrExpression:
	Expression TOK_OR Expression {
		$$ = ptree_create_node(TOK_OR, 0, NULL, $1, $3, NULL);
	}
	;

AndExpression:
	Expression TOK_AND Expression {
		$$ = ptree_create_node(TOK_AND, 0, NULL, $1, $3, NULL);
	}
	;

NotExpression:
	TOK_NOT Expression {
		$$ = ptree_create_node(TOK_NOT, 0, NULL, $2, NULL, NULL);
	}
	;

CompareExpression:
	Expression TOK_EQ Expression {
		$$ = ptree_create_node(TOK_EQ, 0, NULL, $1, NULL, $3);
	}
	| Expression TOK_EQ TOK_GT Expression {
		$$ = ptree_create_node(TOK_GT, TOK_EQ, NULL, $1, $4, NULL);
	}
	| Expression TOK_EQ TOK_LT Expression {
		$$ = ptree_create_node(TOK_LT, TOK_EQ, NULL, $1, $4, NULL);
	}
	| Expression TOK_EQ TOK_LT TOK_GT Expression {
		$$ = ptree_create_node(TOK_LT, TOK_LT, NULL, NULL, NULL, NULL);
	}
	| Expression TOK_EQ TOK_GT TOK_LT Expression {
		$$ = ptree_create_node(TOK_LT, TOK_LT, NULL, NULL, NULL, NULL);
	}
	| Expression TOK_GT Expression {
		$$ = ptree_create_node(TOK_GT, 0, NULL, $1, $3, NULL);
	}
	| Expression TOK_GT TOK_EQ Expression {
		$$ = ptree_create_node(TOK_GT, TOK_EQ, NULL, $1, $4, NULL);
	}
	| Expression TOK_GT TOK_LT Expression {
		$$ = ptree_create_node(TOK_LT, TOK_GT, NULL, $1, $4, NULL);
	}
	| Expression TOK_GT TOK_LT TOK_EQ Expression {
		$$ = ptree_create_node(TOK_LT, TOK_LT, NULL, NULL, NULL, NULL);
	}
	| Expression TOK_LT Expression {
		$$ = ptree_create_node(TOK_LT, 0, NULL, $1, $3, NULL);
	}
	| Expression TOK_LT TOK_EQ Expression {
		$$ = ptree_create_node(TOK_LT, TOK_EQ, NULL, $1, $4, NULL);
	}
	| Expression TOK_LT TOK_GT Expression {
		$$ = ptree_create_node(TOK_LT, TOK_GT, NULL, $1, $4, NULL);
	}
	| Expression TOK_LT TOK_GT TOK_EQ Expression {
		$$ = ptree_create_node(TOK_LT, TOK_LT, NULL, NULL, NULL, NULL);
	}
	;

/* FIXME: 2 shift/reduce conflicts */
AddExpression:
	Expression TOK_PLUS Expression {
		$$ = ptree_create_node(TOK_PLUS, 0, NULL, $1, $3, NULL);
	}
	| Expression TOK_MINUS Expression {
		$$ = ptree_create_node(TOK_MINUS, 0, NULL, $1, $3, NULL);
	}
	;

MultExpression:
	Expression TOK_ASTERISK Expression {
		$$ = ptree_create_node(TOK_ASTERISK, 0, NULL, $1, $3, NULL);
	}
	| Expression TOK_SLASH Expression {
		$$ = ptree_create_node(TOK_SLASH, 0, NULL, $1, $3, NULL);
	}
	;

PowerExpression:
	Expression TOK_ARROWUP Expression {
		$$ = ptree_create_node(TOK_ARROWUP, 0, NULL, $1, $3, NULL);
	}
	;

Sign:
	TOK_MINUS {
		$$ = TOK_MINUS;
	}
	| TOK_PLUS {
		$$ = TOK_PLUS;
	}
	;

SignedExpression:
	Sign SubExpression %prec PREC_UNARY_PM {
		if ($1 == TOK_MINUS && $2->n_tok_type == TOK_INTEGER) {
			$$ = ptree_create_node(TOK_INTEGER, $2->n_num * -1,
			    NULL, NULL, NULL, NULL);
		} else if ($1 == TOK_PLUS && $2->n_tok_type == TOK_INTEGER) {
			$$ = ptree_create_node(TOK_INTEGER, $2->n_num,
			    NULL, NULL, NULL, NULL);
		} else {
			$$ = ptree_create_node($1, 0, NULL, NULL, $2, NULL);
		}
	}
	;

SubExpression:
	TOK_OPAREN Expression TOK_CPAREN {
		$$ = $2;
	}
	| NotExpression
	| Value
	;

Value:
	ID
	| TOK_ABS TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_ABS, 0, NULL, $3, NULL, NULL);
	}
	| TOK_ASC TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_ASC, 0, NULL, $3, NULL, NULL);
	}
	| TOK_ATN TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_ATN, 0, NULL, $3, NULL, NULL);
	}
	| TOK_CHR_D TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_CHR_D, 0, NULL, $3, NULL, NULL);
	}
	| TOK_COS TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_COS, 0, NULL, $3, NULL, NULL);
	}
	| TOK_EXP TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_EXP, 0, NULL, $3, NULL, NULL);
	}
	| TOK_FN TOK_ID TOK_OPAREN ExpressionList TOK_CPAREN {
		$$ = ptree_create_node(TOK_FN, 0, $2, $4, NULL, NULL);
		free($2);
	}
	| TOK_FRE TOK_OPAREN Value TOK_CPAREN {
		$$ = ptree_create_node(TOK_FRE, 0, NULL, $3, NULL, NULL);
	}
	| TOK_INT TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_INT, 0, NULL, $3, NULL, NULL);
	}
	| TOK_LEFT_D TOK_OPAREN Expression TOK_COMMA Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_LEFT_D, 0, NULL, $3, $5, NULL);
	}
	| TOK_LEN TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_LEN, 0, NULL, $3, NULL, NULL);
	}
	| TOK_LOG TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_LOG, 0, NULL, $3, NULL, NULL);
	}
	| TOK_MID_D TOK_OPAREN Expression TOK_COMMA Expression TOK_COMMA Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_MID_D, 0, NULL, $3, $5, $7);
	}
	| TOK_MID_D TOK_OPAREN Expression TOK_COMMA Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_MID_D, 0, NULL, $3, $5, NULL);
	}
	| TOK_PEEK TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_PEEK, 0, NULL, $3, NULL, NULL);
	}
	| TOK_POS TOK_OPAREN Value TOK_CPAREN {
		$$ = ptree_create_node(TOK_POS, 0, NULL, $3, NULL, NULL);
	}
	| TOK_RIGHT_D TOK_OPAREN Expression TOK_COMMA Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_RIGHT_D, 0, NULL, $3, $5, NULL);
	}
	| TOK_RND TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_RND, 0, NULL, $3, NULL, NULL);
	}
	| TOK_SGN TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_SGN, 0, NULL, $3, NULL, NULL);
	}
	| TOK_SIN TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_SIN, 0, NULL, $3, NULL, NULL);
	}
	| TOK_SPC TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_SPC, 0, NULL, $3, NULL, NULL);
	}
	| TOK_SQR TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_SQR, 0, NULL, $3, NULL, NULL);
	}
	| TOK_STR_D TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_STR_D, 0, NULL, $3, NULL, NULL);
	}
	| TOK_TAB TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_TAB, 0, NULL, $3, NULL, NULL);
	}
	| TOK_TAN TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_TAN, 0, NULL, $3, NULL, NULL);
	}
	| TOK_USR TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_USR, 0, NULL, $3, NULL, NULL);
	}
	| TOK_VAL TOK_OPAREN Expression TOK_CPAREN {
		$$ = ptree_create_node(TOK_VAL, 0, NULL, $3, NULL, NULL);
	}
	| Constant
	;

Constant:
	TOK_INTEGER {
		$$ = ptree_create_node(TOK_INTEGER, $1, NULL, NULL, NULL, NULL);
	}
	| TOK_PI {
		$$ = ptree_create_node(TOK_PI, 0, $1, NULL, NULL, NULL);
		free($1);
	}
	| TOK_REAL {
		$$ = ptree_create_node(TOK_REAL, 0, $1, NULL, NULL, NULL);
		free($1);
	}
	| TOK_STRING {
		$$ = ptree_create_node(TOK_STRING, 0, $1, NULL, NULL, NULL);
		free($1);
	}
	;

%%

static node_t *
ptree_create_node(int tok_type, int num, const char *str, node_t *left,
    node_t *mid, node_t *right)
{
	node_t *n = calloc(1, sizeof(node_t));

	n->n_tok_type = tok_type;
	switch (tok_type) {
	case TOK_AND:
	case TOK_NOT:
	case TOK_OR:
		n->n_val_type = left->n_val_type;
		break;
	case TOK_EQ:
		n->n_val_type = (mid == NULL ? VTYPE_BOOL : VTYPE_NONE);
		break;
	case TOK_GT:
	case TOK_LT:
		n->n_val_type = VTYPE_BOOL;
		break;
	case TOK_ID:
		switch ((enum sym_btype)num) {
		case SBTYPE_INT:
		case SBTYPE_REAL:
			n->n_val_type = VTYPE_NUM;
			break;
		case SBTYPE_STRING:
			n->n_val_type = VTYPE_STRING;
			break;
		default:
			n->n_val_type = VTYPE_NONE;
			break;
		}
		break;
	case TOK_ABS:
	case TOK_ASC:
	case TOK_ARROWUP:
	case TOK_ASTERISK:
	case TOK_ATN:
	case TOK_COS:
	case TOK_EXP:
	case TOK_FN:
	case TOK_FRE:
	case TOK_INT:
	case TOK_INTEGER:
	case TOK_LEN:
	case TOK_LOG:
	case TOK_MINUS:
	case TOK_PEEK:
	case TOK_PI:
	case TOK_POS:
	case TOK_REAL:
	case TOK_RND:
	case TOK_SGN:
	case TOK_SIN:
	case TOK_SLASH:
	case TOK_SQR:
	case TOK_TAN:
	case TOK_VAL:
		n->n_val_type = VTYPE_NUM;
		break;
	case TOK_CHR_D:
		n->n_val_type = VTYPE_CHAR;
		break;
	case TOK_PLUS:
		/* strings might be concatenated with '+' */
		if (mid->n_val_type == VTYPE_STRING ||
		    mid->n_val_type == VTYPE_CHAR) {
			n->n_val_type = VTYPE_STRING;
		} else {
			n->n_val_type = VTYPE_NUM;
		}
		break;
	case TOK_LEFT_D:
	case TOK_MID_D:
	case TOK_RIGHT_D:
	case TOK_STR_D:
	case TOK_STRING:
		n->n_val_type = VTYPE_STRING;
		break;
	default:
		n->n_val_type = VTYPE_NONE;
		break;
	}
	n->n_num = num;
	if (str != NULL)
		n->n_str = strdup(str);
	n->n_lineno = basic_lineno;
	if (left != NULL)
		n->n_left = left;
	if (mid != NULL)
		n->n_mid = mid;
	if (right != NULL)
		n->n_right = right;
	adjust_sym_ctype(n);
	return n;
}

void
ptree_free_tree(node_t *root)
{
	if (root == NULL)
		return;
	if (root->n_next != NULL)
		ptree_free_tree(root->n_next);
	if (root->n_left != NULL)
		ptree_free_tree(root->n_left);
	if (root->n_mid != NULL)
		ptree_free_tree(root->n_mid);
	if (root->n_right != NULL)
		ptree_free_tree(root->n_right);
	if (root->n_str != NULL)
		free(root->n_str);
	free(root);
}

node_t *
ptree_find_token(node_t *n, int tok_type)
{
	node_t *found;

	if (n == NULL)
		return NULL;
	if (n->n_tok_type == tok_type)
		return n;
	found = ptree_find_token(n->n_left, tok_type);
	if (found == NULL)
		found = ptree_find_token(n->n_mid, tok_type);
	if (found == NULL)
		found = ptree_find_token(n->n_right, tok_type);
	if (found == NULL)
		found = ptree_find_token(n->n_next, tok_type);
	return found;
}

static enum sym_btype
ID_to_cvar(char *id, node_t *expr_list)
{
	char c;
	int ary_dim = 0, sbtype;
	node_t *np;
	size_t idlen;

	for (np = expr_list; np != NULL; np = np->n_next)
		ary_dim++;
	idlen = strlen(id);
	c = id[idlen - 1];
	if (c == '%' || c == '$')
		id[idlen - 1] = '\0';
	switch (c) {
	case '%':
		c = 'i';
		sbtype = SBTYPE_INT;
		break;
	case '$':
		c = 's';
		sbtype = SBTYPE_STRING;
		break;
	default:
		if (rflag) {
			c = 'r';
			sbtype = SBTYPE_REAL;
		} else {
			c = 'I';
			sbtype = SBTYPE_INT;
		}
		break;
	}
	if (expr_list != NULL)
		sprintf(IDbuf, "a%d_%c_%s", ary_dim, c, id);
	else
		sprintf(IDbuf, "%c_%s", c, id);
	return sbtype;
}

void
yyerror(const char *s)
{
	cb2c_diag(DIAG_ERR, "%s", s);
}
